package com.cxdn.samples.swair.reservation;

import java.util.Hashtable;

/*
* La classe <code>ReservationManager</code> implemente l'interface 
* <code>ReservationRegistry</code> dont l'objectif consiste  grer
* toutes les rservations en cours dans le processus d'excution 
* courant (application).
* Cette implmentation tend la classe abstraite <code>Hashtable</code>.
*
* @version  1.0
*/
public class ReservationManager extends Hashtable implements 
	ReservationRegistry {

	private static ReservationManager uniqueInstance = new ReservationManager();

	/*
	* Construit une instance de la classe <code>ReservationManager</code>.
	*/
	private ReservationManager() {
		super();
	}

	/*
 	* Renvoie le singleton de la classe <code>ReservationManager</code>.
 	*
 	* @return		ReservationManager le singleton de la classe 
 	*				<code>ReservationManager</code>.
	*/
	public static synchronized ReservationManager getUniqueInstance() {
		if (uniqueInstance == null) {
			uniqueInstance = new ReservationManager();
		}
		return uniqueInstance;
	}

	/*
	* Purge le registre des rservations.
	*/
	public synchronized void clear() {
		super.clear();
	}

	/*
	* Renvoie la table des rservations en cours (clone).
	*
	* @return		Hashtable la table des rservations en cours
	*/
	public synchronized Hashtable getAllReservations() {
	    return (Hashtable)clone();
	}

	/*
	* Renvoie la rservation correspondant  l'identifiant pass en
	* paramtre, si elle existe dans le registre, <b>null</b> sinon.
	*
	* @param		reservationId l'identifiant de la rservation
	*
	* @return		Object la rservation rfrence par l'identifiant 
	*				reservationId; <b>null</b> sinon
	*/
	public synchronized Object getReservation(Object reservationId) {
		return (Object)get(reservationId);
	}

	/*
	* Renvoie <b>true</b> si le registre rfrence des rservations
	* en cours.
	*
	* @return		<code>true</code> si le registre contient des
	*				rservations; <code>false</code> sinon
	*/
	public synchronized boolean isEmpty() {
		if (size()==0) {
			return true;
		}
		return false;
	}

	/*
	* Renvoie <b>true</b> si la rservation passe en paramtre
	* est enregistre.
	*
	* @param		reservation la rservation  vrifier
	*
	* @return		<code>true</code> si la rservation est
	*				enregistre; <code>false</code> sinon
	*/
	public synchronized boolean isReservationRegistered(Object reservation) {
		return containsValue(reservation);
	}

	/*
	* Enregistre une nouvelle rservation.
	*
	* @param		reservationId l'identifiant de la rservation 
	* @param		reservation la rservation
	*
	* @return		Object la nouvelle rservation enregistre
	*
	* @throws		ReservationException lance si une anomalie se
	*				produit durant l'enregistrement de la rservation
	*/
	public synchronized Object newReservation(Object reservationId, Object reservation)
		throws ReservationException {

		try {
			put(reservationId, reservation);
		}
		catch (Exception e) {
			e.printStackTrace(System.out);
			StringBuffer sb = new StringBuffer();
			sb.append("une ");
			sb.append(e.getClass());
			sb.append(" s'est produite durant l'enregistrement de la rservation");
			if ((e.getMessage()!=null) && (!(e.getMessage().equals("")))) {
				sb.append(" : ");
				sb.append(e.getMessage());
			}
			throw new ReservationException(sb.toString());
		}
		return reservation;
	}

	/*
	* Libre les rservations du registre.
	*/
	public synchronized void release() {
		clear();
	}

	/*
	* Supprime la rservation identifie du registre.
	*
	* @param		reservationId l'identifiant de la rservation
	*/
	public synchronized void removeReservation(Object reservationId) {
		remove(reservationId);
	}

	/*
	* Renvoie une reprsentation textuelle du registre. 
	*
	* @return		une reprsentation textuelle du registre.
	*/
	public String toString() {
		StringBuffer sb = new StringBuffer();
		sb.append("\n");
		sb.append(getClass().getName());
		sb.append("@");
		sb.append(Integer.toHexString(hashCode()));
		return sb.toString();
	}
}